--------
-- Reads weather conditions and converts them into a bunch of very simple to interpret values for the rest of the script.
--------

-- Global table with converted weather values
local defaultWeatherDefinition = {
  fog = 0,              -- how foggy is it, from 0 to 1
  clear = 1,            -- how clear is the sky, turns to grey with 0
  clouds = 0,           -- how many shapy clouds are up there
  cloudsDensity = 0,    -- how dense are clouds
  tint = rgb(1, 1, 1),  -- color tint for light
  ambi_mod = 0.0,       -- scale tint for high sunangles (>15°)
  saturation = 1,       -- how saturated are the colors
  thunder = 0,          -- lightning amount
  pollution = 0,        -- near fog amount
  pollution_tint = rgb(1, 1, 1),        -- near fog amount
  overcast = 0,         -- overcast modifier
  badness = 0,          -- badness definition
  snow_amount = 0,      -- snow integration
  snow_size = 0,
  snow_rain_mix = 0,
  ash_amount = 0,       -- ash integration
  ash_size = 0,
}

CurrentConditions = table.assign({
  windDir = vec2(0, 0),        -- smoothed wind direction (for clouds)
  windDirInstant = vec2(0, 0), -- normalized wind direction
  windSpeed = 0,               -- smoothed wind speed in m/s (for clouds)
  windSpeedInstant = 0,        -- wind speed in m/s
  rain = 0,
  wetness = 0,
  water = 0,
}, defaultWeatherDefinition, {tint = rgb(1, 1, 1), pollution_tint = rgb(1, 1, 1)})

PreConditions = table.assign({
  windDir = vec2(0, 0),        -- smoothed wind direction (for clouds)
  windDirInstant = vec2(0, 0), -- normalized wind direction
  windSpeed = 0,               -- smoothed wind speed in m/s (for clouds)
  windSpeedInstant = 0,        -- wind speed in m/s
  rain = 0,
  wetness = 0,
  water = 0,
}, defaultWeatherDefinition, {tint = rgb(1, 1, 1), pollution_tint = rgb(1, 1, 1)})

-- Creates and fills table with weather definitions
local weatherDefinitions = {}
ac.WeatherType.NoClouds = 100  -- new type of weather added by Sol and used by Sol controller

local function defineWeather(params)
  weatherDefinitions[params.type] = table.assign({}, defaultWeatherDefinition, params)
end

function WEATHER__get_weather_defintion(id, parameter)
  if weatherDefinitions and weatherDefinitions[id] and weatherDefinitions[id][parameter] then
    return weatherDefinitions[id][parameter]
  end
  return nil
end

defineWeather{type = ac.WeatherType.NoClouds,          fog = 0,   clear = 1,    clouds = 0}
defineWeather{type = ac.WeatherType.Clear,             fog = 0,   clear = 1,    clouds = 0.01 }

defineWeather{type = ac.WeatherType.FewClouds,         fog = 0,   clear = 1,    clouds = 0.10 }
defineWeather{type = ac.WeatherType.ScatteredClouds,   fog = 0,   clear = 1,    clouds = 0.25, cloudsDensity = 0.1 }
defineWeather{type = ac.WeatherType.Windy,             fog = 0,   clear = 0.95, clouds = 0.50, saturation = 0.85}
defineWeather{type = ac.WeatherType.BrokenClouds,      fog = 0,   clear = 0.90, clouds = 0.80, cloudsDensity = 0.2, overcast = 0.3, saturation = 0.70}

defineWeather{type = ac.WeatherType.OvercastClouds,    fog = 0,   clear = 0.2,  clouds = 0.90, cloudsDensity = 0.0, saturation = 0.0, tint = rgb(1.15, 1.09, 1.12), pollution_tint = rgb(0.4, 0.75, 1.35), ambi_mod = 0.5, overcast = 0.8}

defineWeather{type = ac.WeatherType.Cold,              fog = 0.3, clear = 0.9,  clouds = 0.4, saturation = 0.5, tint = rgb(0.8, 0.9, 1.0)}
defineWeather{type = ac.WeatherType.Hot,               fog = 0.1, clear = 1,    clouds = 0.1, saturation = 1.2, tint = rgb(1.0, 0.9, 0.8)}

defineWeather{type = ac.WeatherType.Fog,               fog = 1,    clear = 0.10,  clouds = 0.0, tint = rgb(0.8, 0.87, 1.0), badness = 0.5}
defineWeather{type = ac.WeatherType.Mist,              fog = 0.70, clear = 0.60,  clouds = 0.0, tint = rgb(0.8, 0.9, 1.0), pollution = 0.4, badness = 0.2}
defineWeather{type = ac.WeatherType.Haze,              fog = 0.20, clear = 0.65,  clouds = 0.2, tint = rgb(1, 0.92, 0.9), saturation = 0.5, pollution = 0.0}

defineWeather{type = ac.WeatherType.Dust,              fog = 0.20, clear = 0.70,  clouds = 0.1, tint = rgb(1, 0.85, 0.8), saturation = 1.0, pollution = 0.4, pollution_tint = rgb(1.15, 0.80, 0.50)}
defineWeather{type = ac.WeatherType.Smoke,             fog = 0.50, clear = 0.45,  clouds = 0.2, tint = rgb(0.8, 0.75, 0.7):scale(0.7), ambi_mod = -0.1, saturation = 0.9, pollution = 0.8, pollution_tint = rgb(1, 0.85, 0.80), badness = 0.7, ash_amount = 0.001, ash_size = 0.05}
defineWeather{type = ac.WeatherType.Sand,              fog = 0.30, clear = 0.60,  clouds = 0.0, tint = rgb(1, 0.6, 0.4), ambi_mod = -0.25, pollution = 0.80, pollution_tint = rgb(1, 0.55, 0.25), badness = 0.6}

defineWeather{type = ac.WeatherType.LightDrizzle,      fog = 0.10, clear = 0.7,  clouds = 0.6, cloudsDensity = 0.2, tint = rgb(1.9, 1.95, 2), overcast = 0.25, saturation = 0.7, badness = 0.125}
defineWeather{type = ac.WeatherType.Drizzle,           fog = 0.30, clear = 0.6,  clouds = 0.8, cloudsDensity = 0.4, tint = rgb(0.9, 0.95, 1.0), pollution_tint = rgb(0.75, 0.90, 1.00), overcast = 0.95, badness = 0.25}
defineWeather{type = ac.WeatherType.HeavyDrizzle,      fog = 0.50, clear = 0.2,  clouds = 0.9, cloudsDensity = 0.6, tint = rgb(0.8, 0.9, 1.0), pollution_tint = rgb(0.75, 0.90, 1.00), overcast = 0.5, badness = 0.5}

defineWeather{type = ac.WeatherType.LightRain,         fog = 0.0,  clear = 0.7,  clouds = 0.6, cloudsDensity = 0.3, tint = rgb(0.9, 0.95, 1.0), pollution_tint = rgb(0.65, 0.80, 1.00), overcast = 0.2, badness = 0.25}
defineWeather{type = ac.WeatherType.Rain,              fog = 0.05, clear = 0.1,  clouds = 0.9, cloudsDensity = 0.5, tint = rgb(0.8, 0.9, 1.0), pollution = 0.30, pollution_tint = rgb(0.65, 0.80, 1.00), overcast = 0.8, badness = 0.5}
defineWeather{type = ac.WeatherType.HeavyRain,         fog = 0.07, clear = 0,    clouds = 1.0, cloudsDensity = 1.0, tint = rgb(0.6, 0.7, 1.0), pollution = 0.40, pollution_tint = rgb(0.65, 0.80, 1.00), overcast = 1.0, badness = 1.0}

defineWeather{type = ac.WeatherType.LightThunderstorm, fog = 0.0,  clear = 0.5,  clouds = 0.7, cloudsDensity = 0.6, tint = rgb(0.6, 0.7, 0.8), thunder = 0.4, overcast = 0.5, badness = 0.6}
defineWeather{type = ac.WeatherType.Thunderstorm,      fog = 0.0,  clear = 0.3,  clouds = 0.8, cloudsDensity = 0.7, tint = rgb(0.5, 0.6, 0.7), thunder = 0.6, overcast = 0.6, badness = 0.8}
defineWeather{type = ac.WeatherType.HeavyThunderstorm, fog = 0.10, clear = 0.1,  clouds = 1.0, cloudsDensity = 0.8, tint = rgb(0.4, 0.5, 0.6), thunder = 0.8, overcast = 0.9, badness = 1.0}

defineWeather{type = ac.WeatherType.LightSnow,         fog = 0.2, clear = 0.4,  clouds = 0.6, cloudsDensity = 0.1, tint = rgb(1.0, 1.0, 1.0), ambi_mod = 0.5, overcast = 0.1, pollution = 0.30, pollution_tint = rgb(1.5, 1.3, 1.1), saturation = 1.0, badness = 0.3, snow_rain_mix = 1.0, snow_amount = 0.1, snow_size = 0.1}
defineWeather{type = ac.WeatherType.Snow,              fog = 0.4, clear = 0.2,  clouds = 0.7, cloudsDensity = 0.2, tint = rgb(1.0, 1.0, 1.0), ambi_mod = 0.5, overcast = 0.2, pollution = 0.60, pollution_tint = rgb(1.5, 1.3, 1.1), saturation = 0.5, badness = 0.5, snow_rain_mix = 1.0, snow_amount = 0.5, snow_size = 0.5}
defineWeather{type = ac.WeatherType.HeavySnow,         fog = 0.9, clear = 0.0,  clouds = 0.8, cloudsDensity = 0.3, tint = rgb(1.0, 1.0, 1.0), ambi_mod = 0.5, overcast = 0.5, pollution = 0.70, pollution_tint = rgb(1.0, 1.0, 1.0), badness = 1.0, snow_rain_mix = 1.0, snow_amount = 1.0, snow_size = 1.0}

defineWeather{type = ac.WeatherType.LightSleet,        fog = 0.1, clear = 0.9,  clouds = 0.7, cloudsDensity = 0.2, tint = rgb(0.6, 0.8, 1.0), saturation = 0.25, badness = 0.4, snow_rain_mix = 0.2, snow_amount = 0.1, snow_size = 0.1}
defineWeather{type = ac.WeatherType.Sleet,             fog = 0.3, clear = 0.7,  clouds = 0.8, cloudsDensity = 0.4, tint = rgb(0.6, 0.8, 1.0), saturation = 0.12, badness = 0.7, snow_rain_mix = 0.35, snow_amount = 0.3, snow_size = 0.1}
defineWeather{type = ac.WeatherType.HeavySleet,        fog = 0.5, clear = 0.5,  clouds = 0.9, cloudsDensity = 0.6, tint = rgb(0.6, 0.8, 1.0), saturation = 0.0, badness = 1.0, snow_rain_mix = 0.5, snow_amount = 0.5, snow_size = 0.1}

defineWeather{type = ac.WeatherType.Squalls,           fog = 0.00, clear = 0.6, clouds = 0.8, cloudsDensity = 0.2, saturation = 0.7, tint = rgb(0.70, 0.80, 1.0), overcast = 0.50, badness = 0.4}
defineWeather{type = ac.WeatherType.Tornado,           fog = 0.15, clear = 0.3, clouds = 0.9, cloudsDensity = 0.5, saturation = 0.5, tint = rgb(0.65, 0.72, 0.9), overcast = 0.70, pollution = 0.1, badness = 1.0}
defineWeather{type = ac.WeatherType.Hurricane,         fog = 0.25, clear = 0.0, clouds = 1.0, cloudsDensity = 0.7, saturation = 0.0, tint = rgb(0.45, 0.56, 0.65), ambi_mod = 0.5, overcast = 1.00, pollution = 0.4, badness = 1.0}

defineWeather{type = ac.WeatherType.Hail,              fog = 0.5, clear = 0,    clouds = 1,   tint = rgb(0.3, 0.24, 0.28):adjustSaturation(0.5), thunder = 1, badness = 0.5}













-- PurePlanner settings for online races
__purePlanner_path = ac.getFolder(ac.FolderID.ACAppsLua).."\\PurePlanner\\"
__Planner_Settings_path = ac.getFolder(ac.FolderID.ExtRoot).."\\config-ext\\PurePlanner\\"
__Planner_Settings_WeatherPresets_path = __Planner_Settings_path.."WeatherPresets\\"
-- file for holding the default weather presets
__SETTINGS__weather_presets_file = __Planner_Settings_WeatherPresets_path.."Default_WeatherPresets.json"
__SETTINGS__weather_presets__defaults=nil
__SETTINGS__weather_presets=nil

-- settings will be generated here:
dofile(__purePlanner_path.."settings\\settings_defaults.lua")

function SETTINGS__weather_presets__load_file(file)

    if file and file~="" and __SETTINGS__weather_presets~=nil and __SETTINGS__weather_presets__defaults~=nil then
        local buffer = io.load(file, nil)
        if buffer and #buffer>2 then
            local tmp = json.decode(buffer)
            if tmp~=nil then
                -- copy the data for every single position, so an older file will not cause a degration of the main table
                for i=1,#tmp do
                    for ii=1,2 do
                        -- always load name_id and index from the defaults
                        __SETTINGS__weather_presets[i][ii] = __SETTINGS__weather_presets__defaults[i][ii]
                    end
                    for ii=3,#tmp[i] do
                        __SETTINGS__weather_presets[i][ii] = tmp[i][ii]
                    end
                end
                built_weather_presets_index_table()
            end
        else
            ac.debug("Can't open file", file)
        end
    end
end
SETTINGS__weather_presets__load_file(__SETTINGS__weather_presets_file)

















function __Pure_CC_define_weather(id, data)
  if weatherDefinitions[id]~=nil then
    for k,v in pairs(data) do
      if weatherDefinitions[id][k]~=nil then
        weatherDefinitions[id][k] = v
      end
    end
  end
end

function __Pure_CC_get_weather_parameter(id, k)
  if weatherDefinitions[id]~=nil then
    if weatherDefinitions[id][k]~=nil then
      return weatherDefinitions[id][k]
    end
  end
end

function __Pure_CC_set_weather_parameter(id, k, v)
  if weatherDefinitions[id]~=nil then
    if weatherDefinitions[id][k]~=nil then
      weatherDefinitions[id][k] = v
    end
  end
end









-- Read conditions and keep them here
local state = ac.getConditionsSet()
local previousType = nil

local humidity_LUT = LUT:new({
  --                                                                       blend dist  exp 
  --       1     2         3     4     5       6       7          8        9     10    11       12    13    14         15
  --index, fog,  clear,    r,    g,    b,      sat,    pollution, badness  dist_mod             dist_color_mod         atmo
  { 0.0,   0.00, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     0.20, 1.00, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.5,   0.00, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     0.35, 1.00, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.6,   0.00, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     0.50, 0.70, 1.00,    1.00, 1.00, 1.00,      0.90 },
  { 0.7,   0.01, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     1.00, 0.65, 1.00,    0.50, 1.00, 1.50,      1.10 },
  { 0.8,   0.02, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     1.15, 0.60, 1.00,    0.00, 1.00, 2.00,      1.15 },
  { 0.9,   0.04, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     1.50, 0.55, 1.00,    0.00, 1.00, 3.00,      1.25 },
  { 1.0,   0.07, 1.00,     1.0,  1.0,  1.0,    0.0,    0.0,       0.0,     2.00, 0.50, 1.00,    0.00, 1.00, 4.00,      1.35 },
})


local smog_LUT = LUT:new({
  --                                                                         blend dist  exp 
  --       1     2         3     4      5        6       7          8        9     10    11       12    13    14         15
  --index, fog,  clear,    r,    g,     b,       sat,    pollution, badness  dist_mod             dist_color_mod         atmo
  { 0.0,   0.00, 1.00,     1.0,  1.00,  1.00,    0.0,    0.0,       0.0,     0.20, 1.00, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.1,   0.03, 1.00,     1.0,  1.00,  1.00,    0.0,    0.0,       0.0,     0.80, 0.80, 1.00,    0.99, 0.95, 1.00,      0.90 },
  { 0.2,   0.10, 1.00,     1.0,  1.00,  1.00,    0.0,    0.0,       0.0,     1.05, 0.70, 1.00,    0.98, 0.94, 1.00,      0.90 },
  { 0.3,   0.20, 1.00,     1.0,  1.00,  0.99,    0.0,    0.1,       0.0,     1.10, 0.60, 1.00,    0.97, 0.92, 1.00,      0.90 },
  { 0.5,   0.20, 1.00,     1.0,  0.99,  0.97,    0.1,    0.2,       0.0,     1.20, 0.50, 1.00,    0.95, 0.90, 1.00,      0.83 },
  { 0.6,   0.20, 1.00,     1.0,  0.98,  0.95,    0.2,    0.4,       0.1,     1.50, 0.40, 1.00,    0.90, 0.80, 0.95,      0.80 },
  { 0.7,   0.20, 1.00,    0.95,  0.97,  0.99,    0.4,    0.8,       0.2,     1.70, 0.30, 1.00,    0.80, 0.70, 0.90,      0.75 },
  { 0.8,   0.20, 1.00,     0.9,  0.93,  0.98,    0.6,    1.0,       0.4,     1.80, 0.20, 1.00,    0.70, 0.60, 0.85,      0.70 },
  { 0.9,   0.40, 1.00,    0.85,  0.89,  0.97,    0.8,    1.0,       0.7,     1.90, 0.10, 1.00,    0.60, 0.50, 0.70,      0.65 },
  { 1.0,   0.70, 1.00,     0.8,  0.85,  0.96,    1.0,    1.0,       1.0,     2.00, 0.05, 1.00,    0.50, 0.40, 0.75,      0.60 },
})


local mist_LUT = LUT:new({
  --                                                                          blend dist  exp 
  --       1     2         3      4      5        6       7          8        9     10    11       12    13    14         15
  --index, fog,  clear,    r,     g,     b,       sat,    pollution, badness  dist_mod             dist_color_mod         atmo
  { 0.0,   0.00, 1.00,     1.00,  1.00,  1.00,    0.0,    0.0,       0.0,     0.20, 1.00, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.1,   0.05, 1.00,     1.00,  1.00,  1.00,    0.0,    0.0,       0.0,     2.00, 1.00, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.2,   0.20, 0.90,     1.00,  1.00,  1.00,    0.0,    0.0,       0.0,     3.00, 0.80, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.4,   0.50, 0.80,     1.00,  1.00,  1.00,    0.0,    0.2,       0.0,     4.00, 0.50, 1.00,    1.00, 1.00, 1.00,      0.85 },
  { 0.6,   0.70, 0.65,     0.92,  0.95,  0.90,    0.0,    0.4,       0.1,     3.00, 0.50, 1.00,    1.00, 1.00, 1.00,      0.90 },
  { 0.7,   0.80, 0.50,     0.60,  0.70,  0.80,    0.0,    0.5,       0.3,     2.50, 0.50, 1.00,    1.00, 1.00, 1.00,      0.95 },
  { 0.8,   0.90, 0.35,     0.30,  0.35,  0.50,    0.0,    0.6,       0.5,     2.25, 0.50, 1.00,    1.00, 1.00, 1.00,      1.00 },
  { 0.9,   0.85, 0.20,     0.20,  0.25,  0.40,    0.0,    0.8,       0.8,     2.00, 0.50, 1.00,    1.00, 1.00, 1.00,      1.05 },
  { 1.0,   1.00, 0.10,     0.10,  0.15,  0.30,    0.0,    1.0,       0.9,     2.00, 0.50, 1.00,    1.00, 1.00, 1.00,      1.10 },
})



local fog_mod = {
  dist_blend    = 1.0,
  dist_distance = 1.0,
  dist_exponent = 1.0,
  dist_color    = rgb(1.0, 1.0, 1.0),
  atmosphere    = 0.0,
}

function Conditions_get_fog_mod()
  return fog_mod
end

local function lerpFog(fog, mix)
  CurrentConditions.fog         = math.lerp(CurrentConditions.fog,  fog[1], mix)
  CurrentConditions.clear       = math.min(CurrentConditions.clear, fog[2])

  CurrentConditions.tint.r      = CurrentConditions.tint.r * math.lerp(1, fog[3], mix)
  CurrentConditions.tint.g      = CurrentConditions.tint.g * math.lerp(1, fog[4], mix)
  CurrentConditions.tint.b      = CurrentConditions.tint.b * math.lerp(1, fog[5], mix)

  CurrentConditions.pollution   = math.lerp(CurrentConditions.pollution,  fog[6], mix)
  CurrentConditions.saturation  = math.max(CurrentConditions.saturation, fog[7])
  CurrentConditions.badness     = math.max(CurrentConditions.badness, fog[8])


  fog_mod.dist_blend    = math.max(fog_mod.dist_blend ,    fog[9])
  fog_mod.dist_distance = math.lerp(fog_mod.dist_distance , fog[10], mix)
  fog_mod.dist_exponent = math.lerp(fog_mod.dist_exponent , fog[11], mix)

  fog_mod.dist_color.r  = math.lerp(fog_mod.dist_color.r ,  fog[12], mix)
  fog_mod.dist_color.g  = math.lerp(fog_mod.dist_color.g ,  fog[13], mix)
  fog_mod.dist_color.b  = math.lerp(fog_mod.dist_color.b ,  fog[14], mix)

  fog_mod.atmosphere    = math.lerp(fog_mod.atmosphere ,   fog[15], mix)
end

local function lerpExtraFogs(dt)

  local fog_index = math.max(CurrentConditions.fog, CurrentConditions.pollution)

  local smog = __PURE__fog_layer.smog
  local smog_result   = smog_LUT:get(smog)
  local smog_mix      = math.min(smog, math.max(0, smog - 0.75 * fog_index) / (1 - 0.5*fog_index))

  fog_mod.dist_blend    = smog_result[9]
  fog_mod.dist_distance = smog_result[10]
  fog_mod.dist_exponent = smog_result[11]
  fog_mod.dist_color:set(1.0, 1.0, 1.0)

  local humid_result  = humidity_LUT:get(__PURE__fog_layer.humidity)
  local humid_mix     = math.min( math.min(1, __PURE__fog_layer.humidity*2) , math.max(0, 1-2*fog_index) )

  lerpFog(humid_result, humid_mix)


  fog_index = math.max(CurrentConditions.fog, CurrentConditions.pollution)

  
  lerpFog(smog_result, smog_mix)


  fog_index = math.max(CurrentConditions.fog, CurrentConditions.pollution)

  local mist = __PURE__fog_layer.mist
  local mist_result   = mist_LUT:get(mist)
  local mist_mix      = math.min(mist, math.max(0, mist - 0.75 * fog_index) / (1 - 0.5*fog_index))
  
  lerpFog(mist_result, mist_mix)
end



-- State connection from Pure Planner app, to receive the data from Pure Planner app
__PURE_CTRL__STATE = PureState:new(nil, true, "PurePlanner_Controller")
local _l_next_fast_change = 0



local _l_lag_multi = 0
function CONDITION__get_lag_Multi()
  return _l_lag_multi
end


function ConditionConverterGetState()
  return state
end


local snow_amount = 0
local snow_size = 0
local snow_rain_mix = 0
local ash_amount = 0
local ash_size = 0


function ReadConditions(dt)
  -- Update existing conditions instead of re-reading them to make garbage collector’s life easier
  local s = state
  ac.getConditionsSetTo(s)

  -- check the connection to Pure Planner App
  __PURE_CTRL__STATE:checkInterface()

  if __PURE_CTRL__STATE:isConnected() then

    if __PURE_CTRL__STATE:getValue('time.fastchange') then
        -- catch a fast time change state and react / give it some time
        _l_next_fast_change = os.clock() + 4
        __PURE_CTRL__STATE:setValue('time.fastchange', false)
    else
        if math.abs(os.clock() - _l_next_fast_change) < 1 then
            if PURE__2DCLOUDS_doInstantFade then
                PURE__2DCLOUDS_doInstantFade()
                clouds__3D__accelerateTransition()
                --ac.debug("call instant fade", os.clock())
                _l_next_fast_change = 0
            end
        end
    end
    
    if __AC_SIM.isOnlineRace then

      if __PURE_CTRL__STATE:getValue('pure.direct') then
          
        if  __AC_SIM.raceSessionType ~= ac.SessionType.Qualify and
            __AC_SIM.raceSessionType ~= ac.SessionType.Race then

            s.variableC 		        = __PURE_CTRL__STATE:getValue("ctrl.version") or 0
        
            s.currentType 		      = __PURE_CTRL__STATE:getValue("weather.current") 	or 100
            s.upcomingType 	        = __PURE_CTRL__STATE:getValue("weather.next") 		or 100
            s.transition 		        = __PURE_CTRL__STATE:getValue("weather.transition") 	or 0.0

            s.humidity 		          = __PURE_CTRL__STATE:getValue("weather.humidity")	or 0.4
            s.variableA 		        = __PURE_CTRL__STATE:getValue("weather.mist")		or 0.0
            
            s.rainIntensity	        = __PURE_CTRL__STATE:getValue("weather.rain.intensity") 	or 0.0
            s.rainWetness		        = __PURE_CTRL__STATE:getValue("weather.rain.wetness") 	or 0.0
            s.rainWater		          = __PURE_CTRL__STATE:getValue("weather.rain.water") 		or 0.0

            s.wind.direction	      = __PURE_CTRL__STATE:getValue("weather.wind.direction") 	or 0.0
            s.wind.speedFrom	      = __PURE_CTRL__STATE:getValue("weather.wind.strength") 	or 0.0
            s.wind.speedTo		      = s.wind.speedFrom + s.wind.speedFrom * 0.25
            
            s.temperatures.road 	  = __PURE_CTRL__STATE:getValue("weather.temp.road") 	or 10.0
            s.temperatures.ambient  = __PURE_CTRL__STATE:getValue("weather.temp.ambient") 	or 20.0
        end
      else

        if s.currentType>=0 and s.upcomingType>=0 then
          if (__PURE__get_config("weather.add_online_properties") or false) then
            __PURE__STATE:setValue("weather.current",     s.currentType)
            __PURE__STATE:setValue("weather.next",        s.upcomingType)
            __PURE__STATE:setValue("weather.transition",  s.transition )

            local set_curr = __SETTINGS__weather_presets_index[s.currentType]
            local set_next = __SETTINGS__weather_presets_index[s.upcomingType]

            s.humidity    = math.lerp(set_curr[9], set_next[9], s.transition)
            s.variableA   = math.lerp(set_curr[10], set_next[10], s.transition)
          end
        end
      end
    else
      __PURE__STATE:setValue("weather.current",     s.currentType)
      __PURE__STATE:setValue("weather.next",        s.upcomingType)
      __PURE__STATE:setValue("weather.transition",  s.transition )
    end
  end


  __PURE__STATE:setValue("rain.intensity", __PURE__rainFX.intensity)
  __PURE__STATE:setValue("rain.wetness", __PURE__rainFX.wetness)
  __PURE__STATE:setValue("rain.water", __PURE__rainFX.water)


  if s.currentType ~= previousType then
    previousType = s.currentType
    ForceRapidUpdates = 100
  end

  -- Update wind
  local cc = PreConditions
  local dir = s.wind.direction * math.pi / 180 + math.perlin(Sim.timestamp / 1.04e5, 3) * 2
  cc.windDirInstant:set(-math.sin(dir), math.cos(dir))
  cc.windSpeedInstant = (s.wind.speedFrom + s.wind.speedTo) / (2 * 3.6)
  -- Mixing definitions and applying them smoothly 
  _l_lag_multi = math.lagMult((not SmoothTransition or os.preciseClock() < 1) and 0 or 0.995, dt)
  

  local vc = weatherDefinitions[s.currentType] or defaultWeatherDefinition
  local vu = weatherDefinitions[s.upcomingType] or defaultWeatherDefinition
  cc.fog = cc.fog + (math.lerp(vc.fog, vu.fog, s.transition) - cc.fog) * _l_lag_multi
  cc.clear = cc.clear + (math.lerp(vc.clear, vu.clear, s.transition) - cc.clear) * _l_lag_multi
  cc.clouds = cc.clouds + (math.lerp(vc.clouds, vu.clouds, s.transition) - cc.clouds) * _l_lag_multi
  cc.cloudsDensity = cc.cloudsDensity + (math.lerp(vc.cloudsDensity, vu.cloudsDensity, s.transition) - cc.cloudsDensity) * _l_lag_multi
  cc.saturation = cc.saturation + (math.lerp(vc.saturation, vu.saturation, s.transition) - cc.saturation) * _l_lag_multi
  cc.thunder = cc.thunder + (math.lerp(vc.thunder, vu.thunder, s.transition) - cc.thunder) * _l_lag_multi
  cc.pollution = cc.pollution + (math.lerp(vc.pollution, vu.pollution, s.transition) - cc.pollution) * _l_lag_multi

  cc.pollution_tint.r = cc.pollution_tint.r + (math.lerp(vc.pollution_tint.r, vu.pollution_tint.r, s.transition) - cc.pollution_tint.r) * _l_lag_multi
  cc.pollution_tint.g = cc.pollution_tint.g + (math.lerp(vc.pollution_tint.g, vu.pollution_tint.g, s.transition) - cc.pollution_tint.g) * _l_lag_multi
  cc.pollution_tint.b = cc.pollution_tint.b + (math.lerp(vc.pollution_tint.b, vu.pollution_tint.b, s.transition) - cc.pollution_tint.b) * _l_lag_multi

  cc.overcast = cc.overcast + (math.lerp(vc.overcast, vu.overcast, s.transition) - cc.overcast) * _l_lag_multi
  cc.badness = cc.badness + (math.lerp(vc.badness, vu.badness, s.transition) - cc.badness) * _l_lag_multi

  cc.snow_amount = cc.snow_amount + (math.lerp(vc.snow_amount, vu.snow_amount, s.transition) - cc.snow_amount) * _l_lag_multi
  cc.snow_size = cc.snow_size + (math.lerp(vc.snow_size, vu.snow_size, s.transition) - cc.snow_size) * _l_lag_multi
  cc.snow_rain_mix = cc.snow_rain_mix + (math.lerp(vc.snow_rain_mix, vu.snow_rain_mix, s.transition) - cc.snow_rain_mix) * _l_lag_multi
  cc.ash_amount = cc.ash_amount + (math.lerp(vc.ash_amount, vu.ash_amount, s.transition) - cc.ash_amount) * _l_lag_multi
  cc.ash_size = cc.ash_size + (math.lerp(vc.ash_size, vu.ash_size, s.transition) - cc.ash_size) * _l_lag_multi

  cc.rain = cc.rain + (s.rainIntensity - cc.rain) * _l_lag_multi
  cc.wetness = cc.wetness + (s.rainWetness - cc.wetness) * _l_lag_multi
  cc.water = cc.water + (s.rainWater - cc.water) * _l_lag_multi
  cc.tint.r = cc.tint.r + (math.lerp(vc.tint.r, vu.tint.r, s.transition) - cc.tint.r) * _l_lag_multi
  cc.tint.g = cc.tint.g + (math.lerp(vc.tint.g, vu.tint.g, s.transition) - cc.tint.g) * _l_lag_multi
  cc.tint.b = cc.tint.b + (math.lerp(vc.tint.b, vu.tint.b, s.transition) - cc.tint.b) * _l_lag_multi
  cc.ambi_mod = cc.ambi_mod + (math.lerp(vc.ambi_mod, vu.ambi_mod, s.transition) - cc.ambi_mod) * _l_lag_multi



--[[  only used for CM LCS options - dev mode !
  if table.nkeys(Overrides) > 1 then
    cc.fog            = Overrides.fog           or cc.fog
    cc.clouds         = Overrides.clouds        or cc.clouds
    cc.cloudsDensity  = Overrides.cloudsDensity or cc.cloudsDensity
    cc.clear          = Overrides.clear         or cc.clear
    cc.saturation     = Overrides.saturation    or cc.saturation
    cc.tint           = Overrides.tint and Overrides.tint:clone() or cc.tint
    cc.ambi_mod     = Overrides.ambi_mod    or cc.ambi_mod
    cc.thunder        = Overrides.thunder       or cc.thunder
    cc.pollution      = Overrides.pollution     or cc.pollution
    cc.overcast       = Overrides.overcast      or cc.overcast
    cc.badness        = Overrides.badness       or cc.badness
    cc.rain           = Overrides.rain          or cc.rain
    cc.wetness        = Overrides.wetness       or cc.wetness
    cc.water          = Overrides.water         or cc.water
  end
]]

  CurrentConditions.fog            = PreConditions.fog           
  CurrentConditions.clouds         = PreConditions.clouds        
  CurrentConditions.cloudsDensity  = PreConditions.cloudsDensity 
  CurrentConditions.clear          = PreConditions.clear         
  CurrentConditions.saturation     = PreConditions.saturation

  CurrentConditions.ambi_mod     = PreConditions.ambi_mod 
  CurrentConditions.tint:set(PreConditions.tint)
  CurrentConditions.ambi_mod      = math.clampN(CurrentConditions.ambi_mod, -1, 10)
  CurrentConditions.ambi_mod      = math.max(CurrentConditions.ambi_mod, math.clamp(snow_amount * 10 * (1 - math.max(PreConditions.snow_amount, PreConditions.clear)), 0, 5))
  CurrentConditions.tint:scale(1 + CurrentConditions.ambi_mod * math.saturateN(__IntD(0,1.5)))

  CurrentConditions.thunder        = PreConditions.thunder       
  CurrentConditions.pollution      = PreConditions.pollution
  CurrentConditions.pollution_tint:set(PreConditions.pollution_tint)
  CurrentConditions.overcast       = PreConditions.overcast      
  CurrentConditions.badness        = PreConditions.badness       
  CurrentConditions.rain           = PreConditions.rain          
  CurrentConditions.wetness        = PreConditions.wetness       
  CurrentConditions.water          = PreConditions.water         

  CurrentConditions.snow_amount    = PreConditions.snow_amount      
  CurrentConditions.snow_size      = PreConditions.snow_size
  CurrentConditions.snow_rain_mix  = PreConditions.snow_rain_mix
  CurrentConditions.ash_amount     = PreConditions.ash_amount
  CurrentConditions.ash_size       = PreConditions.ash_size

  lerpExtraFogs(dt)

  -- check the limits
  CurrentConditions.fog           = math.saturateN(CurrentConditions.fog)
  CurrentConditions.clouds        = math.saturateN(CurrentConditions.clouds)
  CurrentConditions.cloudsDensity = math.saturateN(CurrentConditions.cloudsDensity)
  CurrentConditions.clear         = math.saturateN(CurrentConditions.clear)
  CurrentConditions.saturation    = math.clampN(CurrentConditions.saturation, 0, 10)
  CurrentConditions.ambi_mod      = math.clampN(CurrentConditions.ambi_mod, -1, 10)
  CurrentConditions.thunder       = math.saturateN(CurrentConditions.thunder)
  CurrentConditions.pollution     = math.saturateN(CurrentConditions.pollution)
  CurrentConditions.overcast      = math.saturateN(CurrentConditions.overcast)
  CurrentConditions.badness       = math.saturateN(CurrentConditions.badness)
  CurrentConditions.snow_amount   = math.saturateN(CurrentConditions.snow_amount)
  CurrentConditions.snow_size     = math.saturateN(CurrentConditions.snow_size)
  CurrentConditions.snow_rain_mix = math.saturateN(CurrentConditions.snow_rain_mix)
  CurrentConditions.ash_amount    = math.saturateN(CurrentConditions.ash_amount)
  CurrentConditions.ash_size      = math.saturateN(CurrentConditions.ash_size)
  
  CurrentConditions.windDirInstant:set(cc.windDirInstant)
  CurrentConditions.windSpeedInstant = cc.windSpeedInstant


  __PURE__Temperature.ambient = __PURE__Temperature.ambient*(1-dt) + s.temperatures.ambient*dt
  __PURE__Temperature.road    = __PURE__Temperature.road*(1-dt) + s.temperatures.road*dt
  

  -- SNOW

  if ac.setWeatherParticles~=nil then

    if __PURE__get_config("weather.use_weather_particles")~=false then

      local conf_snow_size = __PURE__get_config("weather.snow.size")
      local conf_ash_size = __PURE__get_config("weather.ash.size")

      snow_amount = CurrentConditions.snow_amount
      snow_size = CurrentConditions.snow_size
      snow_rain_mix = CurrentConditions.snow_rain_mix
      ash_amount = CurrentConditions.ash_amount
      ash_size = CurrentConditions.ash_size
      local rain = CurrentConditions.rain

      local temperature = s.temperatures.ambient

      snow_amount   = math.lerp(math.max(rain, snow_amount), snow_amount, math.saturateN(temperature * 0.34))
      snow_size     = math.lerp(math.max(rain*0.25, snow_size), snow_size, math.saturateN(temperature * 0.34)) * conf_snow_size
      snow_rain_mix = math.max(snow_rain_mix, 1 - math.saturateN(temperature * 0.34)) * math.saturateN(conf_snow_size*10-1)
      
      snow_amount = snow_amount * (1 - math.saturateN(CurrentConditions.windSpeedInstant * 0.015)^0.5)
      local damp = math.saturateN(snow_amount*1000-1) * math.saturateN(conf_snow_size*2-1)
      snow_amount = snow_amount * damp

      ash_size = ash_size * conf_ash_size
      damp = math.saturateN(ash_amount*5000-1) * math.saturateN(conf_ash_size*10-1)
      ash_amount = ash_amount * damp

      ac.setSnowMix(snow_rain_mix, snow_size)
      ac.setWeatherParticles('snow', snow_amount, snow_size)
      ac.setWeatherParticles('ash', ash_amount, ash_size)

      ac.setRainWindscreenDropsMultiplier(1 - 0.9*snow_amount*snow_rain_mix)

      CurrentConditions.rain = CurrentConditions.rain * (1 - snow_rain_mix)^2

      ac.setTrackConditionInput('SNOW', snow_amount)
      ac.setTrackConditionInput('ASH', ash_amount)
    else
      ac.setSnowMix(0, 0)
      ac.setWeatherParticles('snow', 0, 0)
      ac.setWeatherParticles('ash', 0, 0)

      ac.setRainWindscreenDropsMultiplier(1)
    end
  end


  --ApplyTornado(1)

end